/*
   *  Object %name    : CC_Functions.c
   *  State           :  %state%
   *  Creation date   :  Aug 4 17:39:24 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief  Secure Boot implemtation of CC functions
   *
   *  \version 
   *  \author Danb
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

#include "error.h"
#include "DX_VOS_Mem.h"
#include "DX_VOS_Memmap.h"
#include "Private_Host_mng.h"
#include "host_hw_defs.h"
#include "host_op_code.h"
#include "private_host_op_code.h"
#include "gen.h"
#include "SEPDriver.h"
#include "MNG_Functions.h"

/*---------------------
    FUNCTIONS
-----------------------*/


const DxInt8_t mw_version[] = 
{
  'I','N','T','L','.','R','3','.','1','_','3','0','0','4','0','9',0
};



/*
*   @brief The function calculates the HASH of the CA RSA public key , the ECC value of the HASH result and burns the HASH & ECC to the OTP at addresses 0x26 - 0x2E ( see Table 1 : OTP bits description ).
*          This function may be called only on the chip manufacture or device manufacture life states.
*          The functions execution steps:
*         1.	calculate HASH - SHA2 - 256 on { E || N }
*         2.	calculate ECC of the SHA2-256 digest.
*         3.	burn the SHA256 digest and the ECC to the OTP
*
*   @param[in] Keysize[in]  - the keyType(OEM | Sjtag) key size in bytes ( the modulus size ).
*   @param[in] N_ptr[in] - the modulus vector.
*   @param[in] ESize[in]- The public exponent size.
*   @param[in] E[in]- The public exponent
*
*   Returns:  Status of the operation.
*/
DxError_t DX_CC_MNG_SetExtRsaPubKeysHashOtp(
                             DxUint32_t    keyIndex, 
                             DxUint32_t    keySize, 
                             DxUint8_t     *N_ptr,
                             DxUint32_t    E)
{
  /* The return error identifier */
  DxError_t           Error;
  
  /* offset */
  DxUint32_t          sramOffset;
   
  /* read param */
  DxUint32_t          messageParam[4];
  
  /*----------------------------------
      CODE
  ------------------------------------*/

  /* .................. initializing local variables ................... */
  /* ------------------------------------------------------------------- */   
   
  /* initializing the Error to O.K */
  Error = DX_OK;      
                         
	/* lock access to the SEP */
  SEPDriver_Lock();
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare params */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_INTEL_SET_EXT_RSA_PUB_KEY_HASH_OTP_OP_CODE;
  messageParam[1] = keyIndex;
  messageParam[2] = E;
  messageParam[3] = keySize;
  
  /* send params */
  SEPDriver_WriteParamater((DxUint32_t)messageParam ,
                           sizeof(DxUint32_t) * 4 ,
                           sizeof(DxUint32_t) * 4 ,
                           &sramOffset , 
                           DX_FALSE);

  /* send N */
  SEPDriver_WriteParamater((DxUint32_t)N_ptr ,
                           sizeof(DxUint32_t) * 64 ,
                           sizeof(DxUint32_t) * 64 ,
                           &sramOffset , 
                           DX_FALSE);
  
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  Error = SEPDriver_POLL_FOR_REPONSE();
  if(Error != DX_OK)
  {
  	goto end_function_unlock;
  }
                                            
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  Error = SEPDriver_StartIncomingMessage(&sramOffset);
  if(Error != DX_OK)
  {
  	goto end_function_unlock;
  }
   
  /* read opcode + status  */
  SEPDriver_ReadParamater((DxUint32_t)messageParam ,
                          sizeof(DxUint32_t) * 2,
                          sizeof(DxUint32_t) * 2,
                          &sramOffset , 
                          DX_FALSE);
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_INTEL_SET_EXT_RSA_PUB_KEY_HASH_OTP_OP_CODE)
  {
    Error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    goto end_function_unlock;
  }
   
  /* check the status */
  if(messageParam[1] != DX_OK)
  {
    Error = messageParam[1];
    goto end_function_unlock;
  }
                                           
  /* ...................... end of function ................................ */   

end_function_unlock:   

  /* lock access to the SEP */
  SEPDriver_Unlock();

  return Error;                                  

}/* DX_CC_MNG_SetExtRsaPubKeysHashOtp */


/*
*   @brief This function returns the release version
*
*   @param[in] releaseVersion_ptr  - pointer to the string where the version is set
*
*   Returns:  Status of the operation.
*/
DxError_t DX_MNG_GetMWPackageVersion(MwVersionInfo_t  version)
{
  DX_VOS_FastMemCpy(version , (DxUint8_t*)mw_version , sizeof(mw_version));
  return DX_OK;
}


